"""Utilities to construct 2D lattices for gauge simulations.

The ``build_lattice`` function produces a list of links on a 2D square
lattice with periodic or open boundary conditions.  Each link is
represented as ``((x, y), mu)`` where ``mu=0`` corresponds to the x‑direction
and ``mu=1`` corresponds to the y‑direction.
"""

from typing import List, Tuple


def build_lattice(size: int, boundary: str = 'periodic') -> List[Tuple[Tuple[int, int], int]]:
    """Build a 2D lattice of given size and boundary conditions.

    Parameters
    ----------
    size : int
        Linear size of the lattice (number of sites along one dimension).
    boundary : str, optional
        Boundary conditions: 'periodic' (default) or 'open'.

    Returns
    -------
    list of tuple
        A list of links ``[((x,y), mu), ...]`` where ``mu=0`` denotes a link
        in the x‑direction and ``mu=1`` denotes a link in the y‑direction.
    """
    links: List[Tuple[Tuple[int, int], int]] = []
    # Directional offsets for mu=0 (x) and mu=1 (y)
    directions = [(1, 0), (0, 1)]
    for x in range(size):
        for y in range(size):
            for mu, (dx, dy) in enumerate(directions):
                nx = x + dx
                ny = y + dy
                if boundary == 'periodic':
                    nx %= size
                    ny %= size
                else:
                    # For open boundaries skip links that would leave the lattice
                    if not (0 <= nx < size and 0 <= ny < size):
                        continue
                links.append(((x, y), mu))
    return links